/*
 * Function for default error printing.
 */

#include <stdarg.h>
#include <stdio.h>
#include "libfma.h"

/*
 * Use varargs. Note: __VA_ARGS__ was not used due to Windows incompatibility.
 */

/*
 * variable args print with newline appended to end of message
 */
void lf_error_print_nl (const char *fmt, ...)
{
  va_list args;
  int n;
  char s [LF_STRING_LEN];

  va_start (args, fmt);
  n = vsnprintf (s, LF_STRING_LEN, fmt, args);
  va_end (args);
  fprintf (stderr, "%s\n", s);
  if (n >= LF_STRING_LEN) {
    fprintf(stderr, "Previous message was truncated!\n");
  }
}

/*
 * variable args print with no newline appended to end of message
 */
void
lf_error_print(
  const char *fmt,
  ...)
{
  va_list args;
  int n;
  char s [LF_STRING_LEN];

  va_start (args, fmt);
  n = vsnprintf (s, LF_STRING_LEN, fmt, args);
  va_end (args);
  fprintf (stderr, "%s", s);
  if (n >= LF_STRING_LEN) {
    fprintf(stderr, "Previous message was truncated!\n");
  }
}

/*
 * Returns a string formatted with varargs. Caller must free
 * returned string!
 */
char *
lf_error_msg_fmt(
  const char *fmt,
  ...)
{
  va_list args;
  int n;
  char *p;
  int size = LF_STRING_LEN;
  int tries = 0;

  if ((p = malloc(size)) == NULL) {
    return NULL;
  }

  while (tries++ < 2) {
    va_start (args, fmt);
    n = vsnprintf (p, size, fmt, args);
    va_end (args);

    if (n > -1 && n < size) {
      return p;
    } else {
      /* try again with more space */
      size *= 2;
      p = realloc(p, size);   /* times 2 for power of 2 style mallocs */
      if (p  == NULL) {
        return NULL;
      }
    }
  }

  fprintf(stderr, "lf_error_msg_fmt confusion!\n");
  return NULL;  /* should never get here */
}

/*
 * Free string formatted with lf_error_msg_fmt()
 */
void
lf_error_msg_free(
  char *p)
{
  LF_FREE(p);
}
